using Microsoft.UI.Xaml;
using Microsoft.UI.Xaml.Controls;
using NixExampleWinUI.Helpers;
using NixExampleWinUI.ViewModels;
using NixUniversalSDK;
using Windows.ApplicationModel.DataTransfer;
using Windows.ApplicationModel.Resources;
using Windows.UI;

// To learn more about WinUI, the WinUI project structure,
// and more about our project templates, see: http://aka.ms/winui-project-info.

namespace NixExampleWinUI.Views;

/// <summary>
/// Custom view used to display a selected <see cref="IColorData"/> instance from a <see cref="IMeasurementData"/> instance
/// </summary>
public sealed partial class ColorSwatch : UserControl
{
    private static readonly ResourceLoader Loader = ResourceLoader.GetForViewIndependentUse();

    /// <summary>
    /// <see cref="ColorSwatchViewModel"/> for this swatch
    /// </summary>
    public ColorSwatchViewModel ViewModel
    {
        get;
    }

    /// <summary>
    /// <see cref="IMeasurementData"/> for this swatch
    /// </summary>
    public IMeasurementData? MeasurementData
    {
        get => ViewModel.MeasurementData;
        set => ViewModel.MeasurementData = value;
    }

    /// <summary>
    /// Current <see cref="ElementTheme"/> applied for this swatch, based on the color rendered on screen. One of <see cref="ElementTheme.Light"/> or <see cref="ElementTheme.Dark"/>.
    /// </summary>
    public ElementTheme ColorTheme => ViewModel.ColorTheme;

    public ColorSwatch()
    {
        ViewModel = App.GetService<ColorSwatchViewModel>();
        InitializeComponent();
    }

    private async void InfoButton_Click(object sender, RoutedEventArgs e)
    {
        if (MeasurementData is not null)
        {
            var dialog = new InfoListContentDialog(items: MeasurementData.GetMeasurementInfoList())
            {
                XamlRoot = this.XamlRoot,
                Title = Loader.GetString("TitleScanInfo")
            };
            await dialog.ShowAsync();
        }
    }

    /// <summary>
    /// Copies hex triplet for the currently displayed color to the clipboard (based on the sRGB value for the displayed <see cref="IColorData"/>)
    /// </summary>
    private void CopyButton_Click(object sender, RoutedEventArgs e)
    {
        if (ViewModel.UIColor is Color color)
        {
            var hexString = string.Format("{0:x2}{1:x2}{2:x2}", color.R, color.G, color.B);
            var dataPackage = new DataPackage();
            dataPackage.SetText(hexString);
            Clipboard.SetContent(dataPackage);

            if (sender is Button button)
            {
                var flyout = new Flyout
                {
                    XamlRoot = button.XamlRoot,
                    Content = new TextBlock
                    {
                        Text = string.Format(Loader.GetString("MessageHexCopied"), hexString),
                        Style = Application.Current.Resources["BaseTextBlockStyle"] as Style
                    }
                };
                flyout.ShowAt(button);
            }
        }
    }
}
